#!/bin/bash

if [[ $(id -u) -eq 0 ]]; then
    echo "Script is running as root. Run without sudo."
    exit 1  # Quit the script
fi

# Define variables
app_name="Video Downloader"
version="1.0"
install_dir="/opt/$app_name"
executable="Downloader"
desktop_file="$HOME/.local/share/applications/$app_name.desktop"

#Already installed?
if [ -d "$install_dir" ]; then
    echo "App already installed."
    echo "If this was an error, please uninstall the app and try again:"
    echo "To uninstall run UNINSTALL.sh at $install_dir/UNINSTALL.sh"
    exit 2
else
    echo "Directory does not exist."
fi

# Function to display usage instructions
usage() {
    echo "Usage: $0 [options]"
    echo "Options:"
    echo "  -h, --help       Display this help message"
    echo "  -p, --prefix     Specify installation directory (default: $install_dir)"
    echo "  -y, --yes        Automatic yes to prompts"
    echo "  -n, --no-prompt  Run without any user prompts"
    echo "  -v, --verbose    Display verbose output"
    exit 0
}

# Parse command-line arguments
while [[ "$#" -gt 0 ]]; do
    case "$1" in
        -h|--help) usage ;;
        -p|--prefix) install_dir="$2"; shift ;;
        -y|--yes) YES=1 ;;
        -n|--no-prompt) NO_PROMPT=1 ;;
        -v|--verbose) VERBOSE=1 ;;
        *) echo "Unknown option: $1"; usage ;;
    esac
    shift
done

#Already installed?
if command -v ffmpeg &>/dev/null; then
    echo "FFMPEG is already installed."
else
    echo "FFMPEG is not already installed."
    echo "Attempting to install FFMPEG."

    if [ "$YES" = "1" ]; then
        # Detect the package manager and install FFMPEG with no prompt
        if [ -x "$(command -v apt)" ]; then
            echo "Detected apt package manager"
            sudo apt update -y
            sudo apt install -y ffmpeg
        elif [ -x "$(command -v yum)" ]; then
            echo "Detected yum package manager"
            sudo yum update -y
            sudo yum install -y ffmpeg-free
        elif [ -x "$(command -v dnf)" ]; then
            echo "Detected dnf package manager"
            sudo dnf update -y
            sudo dnf install -y ffmpeg-free
        elif [ -x "$(command -v pacman)" ]; then
            echo "Detected pacman package manager"
            sudo pacman -Sy
            sudo pacman -S --noconfirm ffmpeg
        elif [ -x "$(command -v zypper)" ]; then
            echo "Detected zypper package manager"
            sudo zypper update -y
            sudo zypper install -y ffmpeg
        else
            echo "No package manager detected from this list:"
            echo "apt, yum, dnf, pacman, zypper"
            echo "Please install FFMPEG manually or try again."
            exit 3
        fi
    else
        # Detect the package manager and install FFMPEG with prompts
        if [ -x "$(command -v apt)" ]; then
            echo "Detected apt package manager"
            sudo apt update
            sudo apt install ffmpeg
        elif [ -x "$(command -v yum)" ]; then
            echo "Detected yum package manager"
            sudo yum update
            sudo yum install ffmpeg-free
        elif [ -x "$(command -v dnf)" ]; then
            echo "Detected dnf package manager"
            sudo dnf update
            sudo dnf install ffmpeg-free
        elif [ -x "$(command -v pacman)" ]; then
            echo "Detected pacman package manager"
            sudo pacman -Sy
            sudo pacman -S ffmpeg
        elif [ -x "$(command -v zypper)" ]; then
            echo "Detected zypper package manager"
            sudo zypper update
            sudo zypper install ffmpeg
        else
            echo "No package manager detected from this list:"
            echo "apt, yum, dnf, pacman, zypper"
            echo "Please install FFMPEG manually or try again."
            exit 3
        fi
    fi
fi

#Check
if command -v ffmpeg &>/dev/null; then
    echo "FFMPEG is definitely installed."
else
    echo "FFMPEG is not installed."
    echo "Failed to install."
    exit 4
fi

# Create installation directory
sudo mkdir -p "$install_dir"

# Copy everything to install directory
sudo cp -r * "$install_dir"

# Set permissions for the executable
sudo chmod +x "$install_dir/$executable"

# Create a desktop file for the application
cat > "$desktop_file" << EOF
[Desktop Entry]
Name=$app_name
Exec="$install_dir/$executable"
Path=$install_dir
Icon=$install_dir/assets/video downloader.svg
Terminal=false
Type=Application
EOF

# Notify the user
echo "Installation completed."
echo "The application has been installed to: $install_dir"
echo "A desktop shortcut has been created."

# Run the application if user wants to
if [ -z "$NO_PROMPT" ]; then
    read -p "Do you want to run the application now? (Y/n) " choice
    case "$choice" in
        y|Y|'') "$install_dir/$executable" ;;
        n|N) echo "You can run the application later by executing: $install_dir/$executable" ;;
        *) echo "Invalid choice, please run the application manually later." ;;
    esac
fi